L = 500;          % aquifer half-width (m)
S = 0.2;          % storage coefficient (-)
TK = 1e-3 * 3600*24*365;     % hydraulic transmissivity (m^2/yr)
h0 = 10;          % aquifer thickness (m)
qre = 0.1;        % spatially uniform recharge (m/yr)
seas = 0;         % seasonality in recharge from 0 (constant) to 1 (between 0 and 2*Qre)
Qp = 0;           % pumping rate at y = L/2 (m^3/yr and per m parallel to rivers)

K = TK / h0;
Taqu = S * L^2 / TK;        % time scale (yr)
hmax = qre * L^2 / (2*TK);  % maximum water table elevation (m)

dt = 0.001;       % time step (yr); see Exercise 3 for stability criterion
t = 0:dt:5;       % time vector (yr)

dy = 0.05 * L;    % grid size in y-direction (m)
y = -L:dy:L;      % vector of y-coordinates (m)

h = zeros(size(y));         % initial condition for hydraulic head (Method 1)
hu = zeros(size(y));        % initial condition for hydraulic head (Method 2)
hdf = h0 * ones(size(y));   % initial condition for hydraulic head (Method 3)
qex = Qp/dy * ones(size(y)); % auxiliary pumping rate vector (m/yr) accounting for dy
p = zeros(size(y));
p(round(0.5*L/dy)) = 1;     % location of pumping at y = L/2
qex = qex .* p;               % pumping rate vector (m/yr)


figure('position',[100 200 1000 250])
subplot(1,3,1)
hold on, box on
xlabel('L (m)'), ylabel('h (m)')

for i = 1:numel(t)

  if mod(2*t(i),1) == 0       % plot two graphs per year
      plot(y,h)
  end

  qret(i) = qre * (1 + seas*sin(2*pi*t(i)));   % seasonally variable recharge of mean Qre
  TKu = TK/h0 * (h0 + h(2:end-1));             % variable "unconfined" transmissivity

  h(2:end-1) =  h(2:end-1)  + ...              % Method 1
    (qret(i) + TK*(h(1:end-2) - 2*h(2:end-1) + h(3:end))/dy^2 - qex(2:end-1))*dt/S;
  hu(2:end-1) = hu(2:end-1) + ...              % Method 2
    (qret(i) + TKu.*(hu(1:end-2) - 2*hu(2:end-1) + hu(3:end))/dy^2 - qex(2:end-1))*dt/S;
  hdf(2:end-1) = hdf(2:end-1) + ...            % Method 3
   (qret(i)+K/2.*(hdf(1:end-2).^2-2*hdf(2:end-1).^2+hdf(3:end).^2)/dy^2-qex(2:end-1))*dt/S;

  hmaxt(i) = max(h);              % results for constant TK - Method 1
  Q1t(i) = TK * (h(2) - h(1)) / dy;
  Q2t(i) = TK * (h(end-1) - h(end)) / dy;

  hmaxtu(i) = max(hu);            % results for TK = K(h + h0) – Method 2
  Q1tu(i) = TKu(1)   * (hu(2) - hu(1)) / dy;
  Q2tu(i) = TKu(end) * (hu(end-1) - hu(end)) / dy;

  hmaxtdf(i) = max(hdf);          % results for hdf^2 – Method 3
  Q1tdf(i) = TK * (hdf(2) - hdf(1)) / dy;
  Q2tdf(i) = TK * (hdf(end-1) - hdf(end)) / dy;

end


subplot(1,3,2)
hold on, box on
plot(t,2*L*qret,'k-')
plot(t,Q1t,'b-')
plot(t,Q2t,'r-')
plot(t,Q1tu,'b--')
plot(t,Q2tu,'r--')
plot(t,Q1tdf,'b:')
plot(t,Q2tdf,'r:')
legend('2Lq_{re}','Q_1','Q_2')
xlabel('t (yr)'), ylabel('2Lq_{re}, Q_1, Q_2 (m^3/yr/m)')

subplot(1,3,3)
hold on, box on
plot(t,hmaxt,'k-')
plot(t,hmaxtu,'k--')
plot(t,hmaxtdf-h0,'k:')
xlabel('t (yr)'), ylabel('h_{max} (m)')

